<?php
namespace WPHS;

if (!defined('ABSPATH')) { exit; }

final class Admin {

    public static function init(): void {
        add_action('admin_menu', [__CLASS__, 'menu']);
        add_action('admin_init', [__CLASS__, 'handle_post']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'assets']);
    }

    public static function menu(): void {
        add_options_page(
            __('WP HyperSpeed', 'wp-hyperspeed'),
            __('WP HyperSpeed', 'wp-hyperspeed'),
            'manage_options',
            'wp-hyperspeed',
            [__CLASS__, 'render']
        );
    }

    public static function assets(string $hook): void {
        if ($hook !== 'settings_page_wp-hyperspeed') { return; }
        wp_enqueue_style('wphs-admin', WPHS_URL . 'assets/css/admin.css', [], WPHS_VERSION);
        wp_enqueue_script('wphs-admin', WPHS_URL . 'assets/js/admin.js', [], WPHS_VERSION, true);
    }

    /** FREE build: Pro is always locked until Pro module sets this filter to true. */
    private static function is_pro_active(): bool {
        return (bool) apply_filters('wphs_is_pro_active', false);
    }

    private static function upgrade_url(string $utm): string {
        return 'https://wphyperspeed.com/pricing?utm_source=plugin&utm_medium=admin&utm_campaign=' . rawurlencode($utm);
    }

    public static function handle_post(): void {
        if (!is_admin() || !current_user_can('manage_options')) { return; }
        if (empty($_POST['wphs_action'])) { return; }
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'wphs_save_settings')) {
            return;
        }

        $action = sanitize_text_field(wp_unslash($_POST['wphs_action']));

        if ($action === 'save_settings') {
            $new = [
                // Cache
                'cache_enabled' => isset($_POST['cache_enabled']),
                'cache_mobile' => isset($_POST['cache_mobile']),
                'cache_ttl_hours' => max(1, min(168, intval($_POST['cache_ttl_hours'] ?? 12))),

                // Media
                'lazyload_images' => isset($_POST['lazyload_images']),
                'lazyload_iframes' => isset($_POST['lazyload_iframes']),
                'lazyload_exclude_classes' => self::sanitize_multiline($_POST['lazyload_exclude_classes'] ?? ''),
                'lazyload_exclude_url_contains' => self::sanitize_multiline($_POST['lazyload_exclude_url_contains'] ?? ''),
                'video_placeholders' => isset($_POST['video_placeholders']),

                // Tweaks
                'disable_emojis' => isset($_POST['disable_emojis']),
                'disable_embeds' => isset($_POST['disable_embeds']),
            ];
            Settings::update($new);
            add_settings_error('wphs_messages', 'wphs_saved', __('Settings saved.', 'wp-hyperspeed'), 'updated');
            do_action('wphs_purge_cache');
        }

        if ($action === 'purge_cache') {
            do_action('wphs_purge_cache');
            add_settings_error('wphs_messages', 'wphs_purged', __('Cache cleared.', 'wp-hyperspeed'), 'updated');
        }

        if ($action === 'save_license') {
            // In FREE, we only store the key for later use by Pro module.
            $key = sanitize_text_field(wp_unslash($_POST['license_key'] ?? ''));
            update_option('wphs_license_key', trim($key), false);
            add_settings_error('wphs_messages', 'wphs_license_saved', __('License key saved.', 'wp-hyperspeed'), 'updated');
        }
    }

    private static function sanitize_multiline($value): string {
        $value = (string)$value;
        $value = wp_unslash($value);
        $lines = preg_split('/\R/u', $value) ?: [];
        $clean = [];
        foreach ($lines as $line) {
            $line = trim(sanitize_text_field($line));
            if ($line !== '') { $clean[] = $line; }
        }
        return implode("\n", $clean);
    }

    public static function render(): void {
        if (!current_user_can('manage_options')) { return; }
        $settings = Settings::get();
        $tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'dashboard';

        $tabs = [
            'dashboard'      => ['label' => __('Dashboard', 'wp-hyperspeed'), 'pro' => false],
            'cache'          => ['label' => __('Cache', 'wp-hyperspeed'), 'pro' => false],
            'file_opt'       => ['label' => __('File Optimization', 'wp-hyperspeed'), 'pro' => true],
            'media'          => ['label' => __('Media', 'wp-hyperspeed'), 'pro' => false],
            'preload'        => ['label' => __('Preload', 'wp-hyperspeed'), 'pro' => true],
            'advanced_rules' => ['label' => __('Advanced Rules', 'wp-hyperspeed'), 'pro' => true],
            'database'       => ['label' => __('Database', 'wp-hyperspeed'), 'pro' => true],
            'cdn'            => ['label' => __('CDN', 'wp-hyperspeed'), 'pro' => true],
            'heartbeat'      => ['label' => __('Heartbeat', 'wp-hyperspeed'), 'pro' => true],
            'tools'          => ['label' => __('Tools', 'wp-hyperspeed'), 'pro' => true],
            'license'        => ['label' => __('License', 'wp-hyperspeed'), 'pro' => false],
        ];

        echo '<div class="wrap wphs-wrap">';

        // Header with logo
        echo '<div class="wphs-header">';
        echo '<img class="wphs-logo" src="' . esc_url(WPHS_URL . 'assets/img/wphs-logo.png') . '" alt="' . esc_attr__('WP HyperSpeed', 'wp-hyperspeed') . '" />';
        echo '<div class="wphs-header-text">';
        echo '<h1>' . esc_html__('WP HyperSpeed', 'wp-hyperspeed') . '</h1>';
        echo '<div class="wphs-sub">' . esc_html__('Supercharge WordPress speed with caching, LazyLoad, and pro-grade optimizations.', 'wp-hyperspeed') . '</div>';
        echo '</div>';
        echo '</div>';

        settings_errors('wphs_messages');

        echo '<nav class="wphs-tabs">';
        foreach ($tabs as $key => $meta) {
            $class = ($tab === $key) ? 'active' : '';
            $url = esc_url(add_query_arg(['page' => 'wp-hyperspeed', 'tab' => $key], admin_url('options-general.php')));
            $lock = ($meta['pro'] && !self::is_pro_active()) ? '<span class="wphs-lock-pill">🔒 ' . esc_html__('Pro', 'wp-hyperspeed') . '</span>' : '';
            echo '<a class="wphs-tab ' . esc_attr($class) . '" href="' . $url . '">' . esc_html($meta['label']) . $lock . '</a>';
        }
        echo '</nav>';

        echo '<div class="wphs-card">';

        switch ($tab) {
            case 'dashboard': self::render_dashboard($settings); break;
            case 'cache': self::render_cache($settings); break;
            case 'file_opt': self::render_file_optimization(); break;
            case 'media': self::render_media($settings); break;
            case 'preload': self::render_preload(); break;
            case 'advanced_rules': self::render_advanced_rules(); break;
            case 'database': self::render_database(); break;
            case 'cdn': self::render_cdn(); break;
            case 'heartbeat': self::render_heartbeat(); break;
            case 'tools': self::render_tools(); break;
            case 'license': self::render_license(); break;
        }

        echo '</div>';
        echo '</div>';
    }

    private static function nonce_field(string $action = 'save_settings'): void {
        wp_nonce_field('wphs_save_settings');
        echo '<input type="hidden" name="wphs_action" value="' . esc_attr($action) . '" />';
    }

    private static function pro_gate_header(string $title, string $utm): void {
        $is_locked = !self::is_pro_active();
        echo '<div class="wphs-section-head">';
        echo '<h2>' . esc_html($title) . '</h2>';
        if ($is_locked) {
            echo '<div class="wphs-pro-cta">';
            echo '<span class="wphs-lock-badge">🔒 ' . esc_html__('Pro feature', 'wp-hyperspeed') . '</span>';
            echo ' <a class="button button-primary" target="_blank" rel="noopener" href="' . esc_url(self::upgrade_url($utm)) . '">' . esc_html__('Upgrade to Pro', 'wp-hyperspeed') . '</a>';
            echo '</div>';
        }
        echo '</div>';
    }

    private static function locked_wrap_open(): void {
        if (!self::is_pro_active()) {
            echo '<div class="wphs-locked-wrap" aria-disabled="true">';
        }
    }

    private static function locked_wrap_close(): void {
        if (!self::is_pro_active()) {
            echo '<div class="wphs-locked-note">' . esc_html__('These settings are visible in Free, but can only be changed with WP HyperSpeed Pro.', 'wp-hyperspeed') . '</div>';
            echo '</div>';
        }
    }

    private static function render_dashboard(array $s): void {
        echo '<h2>' . esc_html__('Quick actions', 'wp-hyperspeed') . '</h2>';
        echo '<p class="wphs-muted">' . esc_html__('WP HyperSpeed is active. Use the buttons below to manage your cache.', 'wp-hyperspeed') . '</p>';

        echo '<div class="wphs-actions">';
        echo '<form method="post">';
        wp_nonce_field('wphs_save_settings');
        echo '<input type="hidden" name="wphs_action" value="purge_cache" />';
        submit_button(__('Clear Cache', 'wp-hyperspeed'), 'secondary', 'submit', false);
        echo '</form>';
        echo '<a class="button" target="_blank" rel="noopener" href="https://wphyperspeed.com/docs?utm_source=plugin&utm_medium=admin&utm_campaign=docs">' . esc_html__('Documentation', 'wp-hyperspeed') . '</a>';
        echo '</div>';

        echo '<hr />';

        echo '<h2>' . esc_html__('All features', 'wp-hyperspeed') . '</h2>';
        echo '<p class="wphs-muted">' . esc_html__('Free includes the essentials. Pro unlocks advanced optimizations like Critical CSS, Delay JS, CDN, Preload, and more.', 'wp-hyperspeed') . '</p>';

        echo '<div class="wphs-feature-grid">';
        self::feature_pill(__('Page Cache', 'wp-hyperspeed'), false);
        self::feature_pill(__('LazyLoad + Exclusions', 'wp-hyperspeed'), false);
        self::feature_pill(__('Video Placeholders', 'wp-hyperspeed'), false);
        self::feature_pill(__('Disable Emojis/Embeds', 'wp-hyperspeed'), false);
        self::feature_pill(__('Critical CSS', 'wp-hyperspeed'), true);
        self::feature_pill(__('Delay JavaScript', 'wp-hyperspeed'), true);
        self::feature_pill(__('Preload (Sitemaps)', 'wp-hyperspeed'), true);
        self::feature_pill(__('CDN', 'wp-hyperspeed'), true);
        self::feature_pill(__('Database Cleanup', 'wp-hyperspeed'), true);
        self::feature_pill(__('Heartbeat Control', 'wp-hyperspeed'), true);
        echo '</div>';
    }

    private static function feature_pill(string $label, bool $pro): void {
        $locked = ($pro && !self::is_pro_active());
        $cls = 'wphs-pill' . ($locked ? ' locked' : '');
        $suffix = $pro ? ' • ' . __('Pro', 'wp-hyperspeed') : '';
        echo '<div class="' . esc_attr($cls) . '">' . esc_html($label . $suffix) . '</div>';
    }

    private static function render_cache(array $s): void {
        echo '<h2>' . esc_html__('Cache', 'wp-hyperspeed') . '</h2>';
        echo '<form method="post">';
        self::nonce_field('save_settings');

        echo '<table class="form-table" role="presentation"><tbody>';

        echo '<tr><th scope="row">' . esc_html__('Enable page cache', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="cache_enabled" ' . checked(!empty($s['cache_enabled']), true, false) . ' /> ' .
             esc_html__('Cache HTML pages for faster loading.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Cache lifespan', 'wp-hyperspeed') . '</th><td>';
        echo '<input type="number" min="1" max="168" name="cache_ttl_hours" value="' . esc_attr(intval($s['cache_ttl_hours'])) . '" /> ';
        echo '<span class="description">' . esc_html__('Hours (1–168).', 'wp-hyperspeed') . '</span>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Mobile cache', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="cache_mobile" ' . checked(!empty($s['cache_mobile']), true, false) . ' /> ' .
             esc_html__('Use a separate cache for mobile devices.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';

        echo '</tbody></table>';

        submit_button(__('Save Changes', 'wp-hyperspeed'));
        echo '</form>';

        echo '<form method="post" style="margin-top:12px;">';
        wp_nonce_field('wphs_save_settings');
        echo '<input type="hidden" name="wphs_action" value="purge_cache" />';
        submit_button(__('Clear Cache', 'wp-hyperspeed'), 'secondary', 'submit', false);
        echo '</form>';
    }

    private static function render_media(array $s): void {
        echo '<h2>' . esc_html__('Media', 'wp-hyperspeed') . '</h2>';
        echo '<form method="post">';
        self::nonce_field('save_settings');

        echo '<h3>' . esc_html__('LazyLoad', 'wp-hyperspeed') . '</h3>';
        echo '<table class="form-table" role="presentation"><tbody>';

        echo '<tr><th scope="row">' . esc_html__('LazyLoad images', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="lazyload_images" ' . checked(!empty($s['lazyload_images']), true, false) . ' /> ' .
             esc_html__('Add loading="lazy" to images where appropriate.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('LazyLoad iframes', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="lazyload_iframes" ' . checked(!empty($s['lazyload_iframes']), true, false) . ' /> ' .
             esc_html__('Add loading="lazy" to iframes where appropriate.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Exclude by CSS class', 'wp-hyperspeed') . '</th><td>';
        echo '<textarea name="lazyload_exclude_classes" rows="4" class="large-text code">' . esc_textarea($s['lazyload_exclude_classes'] ?? '') . '</textarea>';
        echo '<p class="description">' . esc_html__('One class per line. If an image has any of these classes, it will not be lazyloaded.', 'wp-hyperspeed') . '</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row">' . esc_html__('Exclude by URL contains', 'wp-hyperspeed') . '</th><td>';
        echo '<textarea name="lazyload_exclude_url_contains" rows="4" class="large-text code">' . esc_textarea($s['lazyload_exclude_url_contains'] ?? '') . '</textarea>';
        echo '<p class="description">' . esc_html__('One pattern per line. If an image URL contains a pattern, it will not be lazyloaded.', 'wp-hyperspeed') . '</p>';
        echo '</td></tr>';

        echo '</tbody></table>';

        echo '<h3>' . esc_html__('Videos', 'wp-hyperspeed') . '</h3>';
        echo '<table class="form-table" role="presentation"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Smart video placeholders', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="video_placeholders" ' . checked(!empty($s['video_placeholders']), true, false) . ' /> ' .
             esc_html__('Replace YouTube/Vimeo iframes with a lightweight thumbnail until clicked.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';
        echo '</tbody></table>';

        echo '<h3>' . esc_html__('Tweaks', 'wp-hyperspeed') . '</h3>';
        echo '<table class="form-table" role="presentation"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Disable emojis', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="disable_emojis" ' . checked(!empty($s['disable_emojis']), true, false) . ' /> ' .
             esc_html__('Remove emoji scripts and styles.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Disable embeds', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" name="disable_embeds" ' . checked(!empty($s['disable_embeds']), true, false) . ' /> ' .
             esc_html__('Disable WordPress oEmbed scripts where possible.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';
        echo '</tbody></table>';

        submit_button(__('Save Changes', 'wp-hyperspeed'));
        echo '</form>';

        echo '<hr />';
        self::pro_gate_header(__('LCP & Image Preload', 'wp-hyperspeed'), 'media_lcp');
        self::locked_wrap_open();
        echo '<table class="form-table" role="presentation"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Auto-detect LCP image', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" disabled /> ' . esc_html__('Detect the Largest Contentful Paint image and optimize it.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Preload LCP image', 'wp-hyperspeed') . '</th><td>';
        echo '<label><input type="checkbox" disabled /> ' . esc_html__('Preload the LCP image for faster rendering.', 'wp-hyperspeed') . '</label>';
        echo '</td></tr>';
        echo '</tbody></table>';
        self::locked_wrap_close();
    }

    private static function render_file_optimization(): void {
        self::pro_gate_header(__('File Optimization', 'wp-hyperspeed'), 'file_opt');
        self::locked_wrap_open();

        echo '<p class="wphs-muted">' . esc_html__('Minify, combine and optimize CSS/JS safely. Recommended for improved Core Web Vitals.', 'wp-hyperspeed') . '</p>';

        echo '<h3>' . esc_html__('HTML', 'wp-hyperspeed') . '</h3>';
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Minify HTML', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Remove whitespace to reduce HTML size.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '</tbody></table>';

        echo '<h3>' . esc_html__('CSS', 'wp-hyperspeed') . '</h3>';
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Minify CSS', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Minify CSS files.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Optimize CSS delivery (Critical CSS)', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Generate and inline critical CSS.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '</tbody></table>';

        echo '<h3>' . esc_html__('JavaScript', 'wp-hyperspeed') . '</h3>';
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Minify JavaScript', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Minify JS files.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Delay JavaScript', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Delay non-critical JS until user interaction.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Safe mode for jQuery', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled checked /> ' . esc_html__('Keep jQuery compatible when delaying JS.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_preload(): void {
        self::pro_gate_header(__('Preload', 'wp-hyperspeed'), 'preload');
        self::locked_wrap_open();

        echo '<p class="wphs-muted">' . esc_html__('Preload builds cache automatically so visitors get fast pages from the first visit.', 'wp-hyperspeed') . '</p>';
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Activate preload', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Warm the cache based on your site structure.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Sitemap-based preload', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Use sitemap URLs for preload.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Prefetch DNS requests', 'wp-hyperspeed') . '</th><td><textarea class="large-text code" rows="4" disabled>//fonts.gstatic.com' . "\n" . '//www.googletagmanager.com</textarea></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_advanced_rules(): void {
        self::pro_gate_header(__('Advanced Rules', 'wp-hyperspeed'), 'advanced_rules');
        self::locked_wrap_open();

        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Never cache URL(s)', 'wp-hyperspeed') . '</th><td><textarea class="large-text code" rows="4" disabled>/cart/' . "\n" . '/checkout/</textarea></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Never cache cookies', 'wp-hyperspeed') . '</th><td><textarea class="large-text code" rows="4" disabled>woocommerce_items_in_cart</textarea></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Always purge URL(s)', 'wp-hyperspeed') . '</th><td><textarea class="large-text code" rows="4" disabled>/</textarea></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_database(): void {
        self::pro_gate_header(__('Database', 'wp-hyperspeed'), 'database');
        self::locked_wrap_open();

        echo '<p class="wphs-muted">' . esc_html__('Clean revisions, drafts, transients and optimize tables. Schedule automatic cleanup.', 'wp-hyperspeed') . '</p>';
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Revisions cleanup', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Remove post revisions.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Expired transients', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Remove expired transients.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Optimize tables', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Optimize database tables.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Schedule automatic cleanup', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Run cleanup weekly.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_cdn(): void {
        self::pro_gate_header(__('CDN', 'wp-hyperspeed'), 'cdn');
        self::locked_wrap_open();

        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Enable CDN', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Rewrite static file URLs to your CDN CNAME(s).', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('CDN CNAME(s)', 'wp-hyperspeed') . '</th><td><input type="text" class="regular-text" disabled value="cdn.example.com" /> <button class="button" disabled>' . esc_html__('Add', 'wp-hyperspeed') . '</button></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Exclude from CDN', 'wp-hyperspeed') . '</th><td><textarea class="large-text code" rows="4" disabled>/wp-content/uploads/private/</textarea></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_heartbeat(): void {
        self::pro_gate_header(__('Heartbeat', 'wp-hyperspeed'), 'heartbeat');
        self::locked_wrap_open();

        echo '<p class="wphs-muted">' . esc_html__('Reduce or disable the WordPress Heartbeat API to save server resources.', 'wp-hyperspeed') . '</p>';
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Control Heartbeat', 'wp-hyperspeed') . '</th><td><label><input type="checkbox" disabled /> ' . esc_html__('Enable Heartbeat control.', 'wp-hyperspeed') . '</label></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Backend', 'wp-hyperspeed') . '</th><td><select disabled><option>' . esc_html__('Reduce activity', 'wp-hyperspeed') . '</option><option>' . esc_html__('Disable', 'wp-hyperspeed') . '</option></select></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Post editor', 'wp-hyperspeed') . '</th><td><select disabled><option>' . esc_html__('Reduce activity', 'wp-hyperspeed') . '</option><option>' . esc_html__('Disable', 'wp-hyperspeed') . '</option></select></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Frontend', 'wp-hyperspeed') . '</th><td><select disabled><option>' . esc_html__('Reduce activity', 'wp-hyperspeed') . '</option><option>' . esc_html__('Disable', 'wp-hyperspeed') . '</option></select></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_tools(): void {
        self::pro_gate_header(__('Tools', 'wp-hyperspeed'), 'tools');
        self::locked_wrap_open();

        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('Export settings', 'wp-hyperspeed') . '</th><td><button class="button button-primary" disabled>' . esc_html__('Download settings', 'wp-hyperspeed') . '</button></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Import settings', 'wp-hyperspeed') . '</th><td><input type="file" disabled /> <button class="button" disabled>' . esc_html__('Upload & import', 'wp-hyperspeed') . '</button></td></tr>';
        echo '<tr><th scope="row">' . esc_html__('Rollback', 'wp-hyperspeed') . '</th><td><button class="button" disabled>' . esc_html__('Reinstall previous version', 'wp-hyperspeed') . '</button></td></tr>';
        echo '</tbody></table>';

        self::locked_wrap_close();
    }

    private static function render_license(): void {
        $key = (string) get_option('wphs_license_key', '');
        $status = \WPHS\Licensing::get_status();
        $is_pro = self::is_pro_active();

        echo '<h2>' . esc_html__('License', 'wp-hyperspeed') . '</h2>';

        if ($is_pro) {
            echo '<div class="notice notice-success inline"><p><strong>' . esc_html__('Pro is active.', 'wp-hyperspeed') . '</strong> ' . esc_html($status['message']) . '</p></div>';
        } else {
            $msg = $status['message'] ? $status['message'] : __('Enter your license key, then click Verify to unlock Pro.', 'wp-hyperspeed');
            echo '<div class="notice notice-info inline"><p><strong>' . esc_html__('Pro is locked.', 'wp-hyperspeed') . '</strong> ' . esc_html($msg) . '</p></div>';
        }

        echo '<form method="post" class="wphs-license-form">';
        self::nonce_field('save_license');
        echo '<table class="form-table"><tbody>';
        echo '<tr><th scope="row">' . esc_html__('License key', 'wp-hyperspeed') . '</th><td>';
        echo '<input type="text" class="regular-text" name="license_key" value="' . esc_attr($key) . '" placeholder="XXXX-XXXX-XXXX-XXXX" />';
        echo '<p class="description">' . esc_html__('Save the key, then verify to activate Pro on this site.', 'wp-hyperspeed') . '</p>';
        echo '</td></tr>';
        echo '</tbody></table>';
        submit_button(__('Save License Key', 'wp-hyperspeed'));
        echo '</form>';

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:10px;">';
        wp_nonce_field('wphs_license_check');
        echo '<input type="hidden" name="action" value="wphs_license_check" />';
        submit_button(__('Verify License', 'wp-hyperspeed'), 'primary', 'submit', false);
        echo '</form>';

        echo '<p style="margin-top:12px;"><a class="button button-secondary" target="_blank" rel="noopener" href="' . esc_url(self::upgrade_url('license')) . '">' . esc_html__('Get WP HyperSpeed Pro', 'wp-hyperspeed') . '</a></p>';

        if (!empty($status['last_check'])) {
            echo '<p class="wphs-muted">' . sprintf(
                esc_html__('Last check: %s', 'wp-hyperspeed'),
                esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), (int) $status['last_check']))
            ) . '</p>';
        }
    }
}
