<?php
namespace WPHS;

if (!defined('ABSPATH')) { exit; }

/**
 * Licensing framework (Free build).
 *
 * - Stores license key + instance ID
 * - Calls remote validation endpoint (API-ready)
 * - Caches results locally
 * - Exposes 'wphs_is_pro_active' filter used by admin UI
 *
 * Change endpoint via filter 'wphs_license_api_base'.
 */
final class Licensing {

    const OPTION_KEY             = 'wphs_license_key';
    const OPTION_INSTANCE_ID     = 'wphs_instance_id';
    const OPTION_LAST_CHECK      = 'wphs_license_last_check';
    const OPTION_LAST_GOOD_UNTIL = 'wphs_license_last_good_until';
    const OPTION_LAST_STATUS     = 'wphs_license_last_status';
    const OPTION_LAST_MESSAGE    = 'wphs_license_last_message';

    const CHECK_TTL = 12 * HOUR_IN_SECONDS;

    public static function init(): void {
        add_filter('wphs_is_pro_active', [__CLASS__, 'filter_is_pro_active']);
        add_action('admin_post_wphs_license_check', [__CLASS__, 'handle_manual_check']);
        add_action('wphs_license_cron', [__CLASS__, 'maybe_check_license']);
        register_activation_hook(WPHS_PATH . 'wp-hyperspeed.php', [__CLASS__, 'on_activate']);
        register_deactivation_hook(WPHS_PATH . 'wp-hyperspeed.php', [__CLASS__, 'on_deactivate']);
    }

    public static function on_activate(): void {
        self::ensure_instance_id();
        if (!wp_next_scheduled('wphs_license_cron')) {
            wp_schedule_event(time() + 300, 'twicedaily', 'wphs_license_cron');
        }
    }

    public static function on_deactivate(): void {
        $ts = wp_next_scheduled('wphs_license_cron');
        if ($ts) {
            wp_unschedule_event($ts, 'wphs_license_cron');
        }
    }

    public static function ensure_instance_id(): string {
        $id = (string) get_option(self::OPTION_INSTANCE_ID, '');
        if (!$id) {
            $id = wp_generate_uuid4();
            update_option(self::OPTION_INSTANCE_ID, $id, false);
        }
        return $id;
    }

    public static function get_key(): string {
        return (string) get_option(self::OPTION_KEY, '');
    }

    public static function get_status(): array {
        return [
            'last_check' => (int) get_option(self::OPTION_LAST_CHECK, 0),
            'good_until' => (int) get_option(self::OPTION_LAST_GOOD_UNTIL, 0),
            'status'     => (string) get_option(self::OPTION_LAST_STATUS, ''),
            'message'    => (string) get_option(self::OPTION_LAST_MESSAGE, ''),
        ];
    }

    public static function filter_is_pro_active(bool $is_pro): bool {
        $good_until = (int) get_option(self::OPTION_LAST_GOOD_UNTIL, 0);
        return ($good_until > time());
    }

    public static function handle_manual_check(): void {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have permission.', 'wp-hyperspeed'));
        }
        check_admin_referer('wphs_license_check');

        self::check_license(true);

        wp_safe_redirect(add_query_arg([
            'page' => 'wp-hyperspeed',
            'tab'  => 'license',
        ], admin_url('options-general.php')));
        exit;
    }

    public static function maybe_check_license(): void {
        self::check_license(false);
    }

    public static function check_license(bool $force = false): void {
        $key = trim(self::get_key());
        if ($key === '') {
            update_option(self::OPTION_LAST_STATUS, 'missing', false);
            update_option(self::OPTION_LAST_MESSAGE, __('No license key set.', 'wp-hyperspeed'), false);
            update_option(self::OPTION_LAST_GOOD_UNTIL, 0, false);
            return;
        }

        $last = (int) get_option(self::OPTION_LAST_CHECK, 0);
        if (!$force && $last && (time() - $last) < self::CHECK_TTL) {
            return;
        }

        $api_base = (string) apply_filters('wphs_license_api_base', 'https://license.wphyperspeed.com/wp-json/wphs/v1');
        $endpoint = rtrim($api_base, '/') . '/license/verify';

        $payload = [
            'license_key' => $key,
            'instance_id' => self::ensure_instance_id(),
            'site_url'    => home_url('/'),
            'plugin'      => 'wp-hyperspeed',
            'version'     => WPHS_VERSION,
        ];

        $resp = wp_remote_post($endpoint, [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body'    => wp_json_encode($payload),
        ]);

        update_option(self::OPTION_LAST_CHECK, time(), false);

        if (is_wp_error($resp)) {
            update_option(self::OPTION_LAST_STATUS, 'error', false);
            update_option(self::OPTION_LAST_MESSAGE, $resp->get_error_message(), false);
            return;
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        if ($code !== 200 || !is_array($data)) {
            update_option(self::OPTION_LAST_STATUS, 'error', false);
            update_option(self::OPTION_LAST_MESSAGE, __('Unexpected response from license server.', 'wp-hyperspeed'), false);
            return;
        }

        $valid = !empty($data['valid']);
        $message = isset($data['message']) ? sanitize_text_field((string) $data['message']) : '';
        $good_until = isset($data['good_until']) ? (int) $data['good_until'] : 0;

        if ($valid && $good_until > time()) {
            update_option(self::OPTION_LAST_STATUS, 'valid', false);
            update_option(self::OPTION_LAST_MESSAGE, $message ?: __('License active.', 'wp-hyperspeed'), false);
            update_option(self::OPTION_LAST_GOOD_UNTIL, $good_until, false);
        } else {
            update_option(self::OPTION_LAST_STATUS, 'invalid', false);
            update_option(self::OPTION_LAST_MESSAGE, $message ?: __('License invalid or expired.', 'wp-hyperspeed'), false);
            update_option(self::OPTION_LAST_GOOD_UNTIL, 0, false);
        }
    }
}
