<?php
namespace WPHS\Features;

use WPHS\Settings;

if (!defined('ABSPATH')) { exit; }

final class LazyLoad {

    public static function init(): void {
        add_filter('the_content', [__CLASS__, 'filter_content'], 20);
        add_filter('widget_text', [__CLASS__, 'filter_content'], 20);
        add_filter('render_block', [__CLASS__, 'filter_content'], 20, 1);
    }

    public static function filter_content($content) {
        if (!is_string($content) || $content === '') { return $content; }

        $s = Settings::get();

        $do_imgs = !empty($s['lazyload_images']);
        $do_iframes = !empty($s['lazyload_iframes']);

        if (!$do_imgs && !$do_iframes) { return $content; }

        // Quick bail-out if no tags found
        if (($do_imgs && stripos($content, '<img') === false) && ($do_iframes && stripos($content, '<iframe') === false)) {
            return $content;
        }

        $exclude_classes = self::lines($s['lazyload_exclude_classes'] ?? '');
        $exclude_url_contains = self::lines($s['lazyload_exclude_url_contains'] ?? '');

        // Use DOMDocument for safer parsing
        $libxml_prev = libxml_use_internal_errors(true);
        $dom = new \DOMDocument();
        $wrapped = '<!DOCTYPE html><html><head><meta charset="utf-8"></head><body>' . $content . '</body></html>';
        $ok = $dom->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        if (!$ok) {
            libxml_clear_errors();
            libxml_use_internal_errors($libxml_prev);
            return $content;
        }

        if ($do_imgs) {
            foreach ($dom->getElementsByTagName('img') as $img) {
                /** @var \DOMElement $img */
                if ($img->hasAttribute('data-wphs-no-lazy')) { continue; }

                $class = $img->getAttribute('class');
                if (self::class_matches($class, $exclude_classes)) { continue; }

                $src = $img->getAttribute('src');
                if (self::url_matches($src, $exclude_url_contains)) { continue; }

                // Skip if already set
                if ($img->hasAttribute('loading')) { continue; }

                $img->setAttribute('loading', 'lazy');
            }
        }

        if ($do_iframes) {
            foreach ($dom->getElementsByTagName('iframe') as $ifr) {
                /** @var \DOMElement $ifr */
                if ($ifr->hasAttribute('data-wphs-no-lazy')) { continue; }
                if ($ifr->hasAttribute('loading')) { continue; }
                $ifr->setAttribute('loading', 'lazy');
            }
        }

        $body = $dom->getElementsByTagName('body')->item(0);
        $new = '';
        if ($body) {
            foreach ($body->childNodes as $child) {
                $new .= $dom->saveHTML($child);
            }
        } else {
            $new = $content;
        }

        libxml_clear_errors();
        libxml_use_internal_errors($libxml_prev);

        return $new;
    }

    private static function lines(string $multiline): array {
        $lines = preg_split('/\R/u', (string)$multiline) ?: [];
        $out = [];
        foreach ($lines as $l) {
            $l = trim($l);
            if ($l !== '') { $out[] = $l; }
        }
        return $out;
    }

    private static function class_matches(string $class_attr, array $exclude): bool {
        if ($class_attr === '' || empty($exclude)) { return false; }
        $classes = preg_split('/\s+/', trim($class_attr)) ?: [];
        $classes = array_filter($classes);
        foreach ($classes as $c) {
            foreach ($exclude as $ex) {
                if ($ex !== '' && $c === $ex) { return true; }
            }
        }
        return false;
    }

    private static function url_matches(string $url, array $patterns): bool {
        if ($url === '' || empty($patterns)) { return false; }
        foreach ($patterns as $p) {
            if ($p !== '' && stripos($url, $p) !== false) { return true; }
        }
        return false;
    }
}
