<?php
namespace WPHS\Features;

use WPHS\Settings;

if (!defined('ABSPATH')) { exit; }

final class PageCache {

    private static $cache_dir = '';

    public static function init(): void {
        self::$cache_dir = WP_CONTENT_DIR . '/cache/wp-hyperspeed';

        // Serve cache as early as possible (still within WP lifecycle for MVP).
        add_action('template_redirect', [__CLASS__, 'maybe_serve_cache'], 0);

        // Generate cache for eligible requests.
        add_action('template_redirect', [__CLASS__, 'start_buffer'], 1);

        // Purge hook
        add_action('wphs_purge_cache', [__CLASS__, 'purge_all']);

        // Purge on content changes
        add_action('save_post', [__CLASS__, 'purge_all']);
        add_action('deleted_post', [__CLASS__, 'purge_all']);
        add_action('switch_theme', [__CLASS__, 'purge_all']);
    }

    public static function maybe_serve_cache(): void {
        $s = Settings::get();
        if (empty($s['cache_enabled'])) { return; }
        if (!self::is_cacheable_request($s)) { return; }

        $file = self::cache_file_path($s);
        if (!$file || !file_exists($file)) { return; }

        // TTL check
        $ttl = max(1, min(168, intval($s['cache_ttl_hours'] ?? 12))) * HOUR_IN_SECONDS;
        $mtime = @filemtime($file);
        if ($mtime && (time() - $mtime) > $ttl) {
            @unlink($file);
            return;
        }

        header('X-WP-HyperSpeed-Cache: HIT');
        header('Content-Type: text/html; charset=' . get_bloginfo('charset'));
        readfile($file);
        exit;
    }

    public static function start_buffer(): void {
        $s = Settings::get();
        if (empty($s['cache_enabled'])) { return; }
        if (!self::is_cacheable_request($s)) { return; }

        ob_start(function($html) use ($s) {
            if (!is_string($html) || $html === '') { return $html; }

            // Only cache successful HTML
            if (http_response_code() !== 200) { return $html; }
            if (stripos($html, '<html') === false) { return $html; }

            $file = self::cache_file_path($s);
            if (!$file) { return $html; }

            self::ensure_dir(dirname($file));
            @file_put_contents($file, $html, LOCK_EX);

            header('X-WP-HyperSpeed-Cache: MISS');
            return $html;
        });
    }

    private static function is_cacheable_request(array $s): bool {
        if (!empty($_GET) || !empty($_POST)) { return false; } // keep simple for MVP
        if (is_user_logged_in()) { return false; }
        if (is_admin()) { return false; }
        if (defined('DOING_AJAX') && DOING_AJAX) { return false; }
        if (defined('REST_REQUEST') && REST_REQUEST) { return false; }
        if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'GET') { return false; }
        if (is_preview()) { return false; }
        if (function_exists('is_cart') && is_cart()) { return false; }
        if (function_exists('is_checkout') && is_checkout()) { return false; }
        if (function_exists('is_account_page') && is_account_page()) { return false; }
        return true;
    }

    private static function cache_file_path(array $s): ?string {
        $url = home_url(add_query_arg([], $_SERVER['REQUEST_URI'] ?? '/'));
        $is_mobile = (!empty($s['cache_mobile']) && wp_is_mobile()) ? 'm' : 'd';
        $key = md5($url . '|' . $is_mobile);

        // Group by host to avoid clashes on multisite-ish envs
        $host = wp_parse_url(home_url(), PHP_URL_HOST);
        if (!$host) { $host = 'site'; }

        return trailingslashit(self::$cache_dir) . sanitize_key($host) . '/' . $key . '.html';
    }

    private static function ensure_dir(string $dir): void {
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }
    }

    public static function purge_all(): void {
        $dir = self::$cache_dir;
        if (!is_dir($dir)) { return; }
        self::rrmdir($dir);
        self::ensure_dir($dir);
    }

    private static function rrmdir(string $dir): void {
        $items = @scandir($dir);
        if (!$items) { return; }
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') { continue; }
            $path = $dir . DIRECTORY_SEPARATOR . $item;
            if (is_dir($path)) {
                self::rrmdir($path);
                @rmdir($path);
            } else {
                @unlink($path);
            }
        }
    }
}
