<?php
namespace WPHS\Features;

use WPHS\Settings;

if (!defined('ABSPATH')) { exit; }

final class VideoPlaceholders {

    public static function init(): void {
        add_filter('the_content', [__CLASS__, 'filter_content'], 15);
        add_action('wp_enqueue_scripts', [__CLASS__, 'assets']);
    }

    public static function assets(): void {
        $s = Settings::get();
        if (empty($s['video_placeholders'])) { return; }
        wp_enqueue_style('wphs-video', WPHS_URL . 'assets/css/video.css', [], WPHS_VERSION);
        wp_enqueue_script('wphs-video', WPHS_URL . 'assets/js/video.js', [], WPHS_VERSION, true);
    }

    public static function filter_content($content) {
        if (!is_string($content) || $content === '') { return $content; }
        $s = Settings::get();
        if (empty($s['video_placeholders'])) { return $content; }
        if (stripos($content, '<iframe') === false) { return $content; }

        $libxml_prev = libxml_use_internal_errors(true);
        $dom = new \DOMDocument();
        $wrapped = '<!DOCTYPE html><html><head><meta charset="utf-8"></head><body>' . $content . '</body></html>';
        $ok = $dom->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        if (!$ok) {
            libxml_clear_errors();
            libxml_use_internal_errors($libxml_prev);
            return $content;
        }

        $iframes = $dom->getElementsByTagName('iframe');
        // DOMNodeList is live; collect first
        $to_process = [];
        foreach ($iframes as $ifr) { $to_process[] = $ifr; }

        foreach ($to_process as $ifr) {
            /** @var \DOMElement $ifr */
            $src = $ifr->getAttribute('src');
            if (!$src) { continue; }

            $info = self::extract_provider($src);
            if (!$info) { continue; }

            $provider = $info['provider'];
            $id = $info['id'];

            $thumb = '';
            if ($provider === 'youtube') {
                $thumb = 'https://i.ytimg.com/vi/' . rawurlencode($id) . '/hqdefault.jpg';
            } elseif ($provider === 'vimeo') {
                // Vimeo thumbnails require API; we fallback to a generic placeholder
                $thumb = '';
            }

            $div = $dom->createElement('div');
            $div->setAttribute('class', 'wphs-video-placeholder');
            $div->setAttribute('data-provider', $provider);
            $div->setAttribute('data-video-id', $id);

            // Preserve iframe attributes (title/allow/etc.) for the player load
            $div->setAttribute('data-iframe-src', $src);

            $button = $dom->createElement('button', '▶');
            $button->setAttribute('type', 'button');
            $button->setAttribute('class', 'wphs-play');

            if ($thumb) {
                $img = $dom->createElement('img');
                $img->setAttribute('class', 'wphs-thumb');
                $img->setAttribute('alt', '');
                $img->setAttribute('loading', 'lazy');
                $img->setAttribute('src', $thumb);
                $div->appendChild($img);
            } else {
                $fallback = $dom->createElement('div');
                $fallback->setAttribute('class', 'wphs-thumb-fallback');
                $fallback->nodeValue = strtoupper($provider) . ' VIDEO';
                $div->appendChild($fallback);
            }

            $div->appendChild($button);

            // Replace iframe with placeholder
            $ifr->parentNode->replaceChild($div, $ifr);
        }

        $body = $dom->getElementsByTagName('body')->item(0);
        $new = '';
        if ($body) {
            foreach ($body->childNodes as $child) {
                $new .= $dom->saveHTML($child);
            }
        } else {
            $new = $content;
        }

        libxml_clear_errors();
        libxml_use_internal_errors($libxml_prev);

        return $new;
    }

    private static function extract_provider(string $src): ?array {
        $u = wp_parse_url($src);
        if (!is_array($u) || empty($u['host'])) { return null; }
        $host = strtolower($u['host']);

        // YouTube
        if (strpos($host, 'youtube.com') !== false || strpos($host, 'youtu.be') !== false) {
            // Patterns:
            // youtube.com/embed/VIDEOID
            // youtu.be/VIDEOID
            $path = $u['path'] ?? '';
            $path = trim($path, '/');
            $id = '';

            if (strpos($host, 'youtu.be') !== false) {
                $id = $path;
            } else {
                // embed/ID
                if (strpos($path, 'embed/') === 0) {
                    $id = substr($path, 6);
                }
            }

            if ($id) {
                $id = preg_replace('/[^a-zA-Z0-9_\-]/', '', $id);
                return ['provider' => 'youtube', 'id' => $id];
            }
        }

        // Vimeo
        if (strpos($host, 'vimeo.com') !== false) {
            // player.vimeo.com/video/ID
            $path = trim(($u['path'] ?? ''), '/');
            $id = '';
            if (preg_match('~video/(\d+)~', $path, $m)) {
                $id = $m[1];
            } elseif (preg_match('~^(\d+)$~', $path, $m)) {
                $id = $m[1];
            }
            if ($id) {
                return ['provider' => 'vimeo', 'id' => $id];
            }
        }

        return null;
    }
}
